
#define _GNU_SOURCE

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <ctype.h>
#include <string.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <dlfcn.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include <semaphore.h>
#include "makepath.h"

/* UNIONS module
 *
 * Preload before libc (and after cwd.so) to perform "union" or
 * "overlay" mounts, where one directory tree appears to become layered
 * on top of another.  Can be used to make CDs "writable", to conveniently
 * seperate source and object files in build trees, to allow package
 * management software to "install" files that actually end up in packages,
 * etc.
 *
 * Depends on cwd.so module, which implements CWD in user space, and
 * translates all pathnames in system calls into absolute paths.
 *
 * NOTE: cwd.so must be loaded before unions.so
 *
 * Compile with:
 *    gcc -nostartfiles -fpic -shared -o unions.so unions.c -ldl
 *
 * System-wide usage:
 *    /etc/ld.so.preload should contain the lines:
 *	/usr/local/lib/cwd.so
 *	/usr/local/lib/unions.so
 *
 *    /etc/unions contains lines like:
 *	overlay:target(options)
 *
 * User or process specific usage:
 *    export LD_PRELOAD=/usr/local/lib/cwd.so:/usr/local/lib/unions.so
 *	(only needed if unions.so doesn't appear in /etc/ld.so.preload)
 *    export UNIONS=overlay@target(options):...
 *
 * For example, export UNIONS=/tmp/cdrom@/mnt/cdrom(create,copyonwrite)
 * causes all the files in /tmp/cdrom to appear overlayed on /mnt/cdrom.
 * Any attempted changes to /mnt/cdrom will actually occur on /tmp/cdrom.
 */

/* #define DEBUG 1 to enable debugging output to stderr */

#define DEBUG 0

/* #define __OVERRIDE 1 to generate aliases for __ varients of 
 * all overridden functions
 */

#define __OVERRIDE 1



#if DEBUG
#define DPRINTF(args) fprintf args
#else
#define DPRINTF(args)
#endif

#if __OVERRIDE
#define __ALIAS(type, syscall, arglist) \
   type __##syscall arglist __attribute__ ((alias( #syscall )))
#else
#define __ALIAS(type, syscall, arglist)
#endif

#define NOTREACHED assert(0)


static int (*nextxstat)(int, const char *, struct stat *);
static int (*nextlxstat)(int, const char *, struct stat *);

static int (*nextopen)(const char *, int, ...);

static int (*nextreadlink)(const char *, char *, size_t);

static DIR * (*nextopendir) (const char * name);
static int (*nextclosedir) (DIR *dirp);
static struct dirent * (*nextreaddir) (DIR *dirp);
static struct dirent64 * (*nextreaddir64) (DIR *dirp);

/* MAY NEED readdir_r (reentrant version of readdir) */

static void (*nextrewinddir)(DIR *dirp);
static void (*nextseekdir)(DIR *dirp, off_t pos);
static off_t (*nexttelldir)(DIR *dirp);

/* scandir MAY NOT BE NEEDED - implemented with __readdir */

struct overlay {
  struct overlay *next;
  char *realname;		/* Must not end with '/' */
  int realnamelen;		/* strlen(realname) stored for efficiency */
  char *mountpoint;		/* Must not end with '/' */
  int mountpointlen;		/* strlen(mountpoint) stored for efficiency */
  int flags;
};

const int CopyOnWrite = 1;
const int CreateHere = 2;

static struct overlay * overlays = NULL;


/* Some special cases that must be considered:
 *
 * cwd.so, when initializing, may need to determine CWD in the "traditional"
 * manner - by opening and stating "..", then "../..", and so on until
 * it reaches the root of the directory tree.  See getcwd() code in glibc,
 * or just strace /bin/pwd to understand this behavior.  This is the only
 * case when these functions may be called with relative pathnames, but
 * that's not a problem since no translations need be done during the
 * initialization.  However, we must be careful to ensure that
 * lstat("/", ...) actually returns a stat structure for the root directory,
 * and not some other directory that might be overlaid on top of it.
 *
 * I handle this by treating directories as a special case.  If the
 * unmodified pathname refers to an existing directory, no overlay
 * expansion is performed upon it.  This ensures that getcwd() will
 * work, even if we've overlaid "/tmp/myroot @ /", for example.
 * open() is handled as an even more special case, but it has to be
 * special, so that getdents() will return a combination of both
 * directory trees and not just one or the other.
 *
 * Symlinks - can exist in an overlay (especially if they were created
 * there), but point to a file in the underlying structure.  cwd.so
 * takes care of this for us when it expands the pathname into
 * absolute form by doing a readlink(), which this module intercepts.
 *
 * Renames - should they be treated as a link followed by an unlink?
 * If so, they should obey the semantics of the "create" and "delete"
 * options.  Currently, they don't.
 *
 * Possible values of "mustexist" argument:
 *
 * NoExistanceCheck     Don't do anything.  Just find the first overlay
 *                      that matches and return the constructed filename.
 *
 * Create               Similar, but creates any directories along the
 *                      path in the overlay.  Doesn't actually create the file.
 *			Returns EEXIST if the file already exists.
 *
 * CreateNoCheck        Similar, but creates any directories along the
 *                      path in the overlay.  Doesn't actually create the file.
 *			Doesn't check if the file exists.
 *
 * LinkMustExist        Matches files, directories, or symlinks.
 *
 * MustExist            Matches files and directories, and follows symlinks.
 *
 * DirMustExist		Only matches directories.
 */

enum mustexist { MustExist, DirMustExist, LinkMustExist,
		 NoExistanceCheck, Create, CreateNoCheck };

/* nextstat(pathname, statptr, overlay, create)
 *
 * The workhorse function for doing overlay lookups.  "pathname"
 * is the virtual (visible) path that we want to convert into
 * actual path(s).  We end up doing a lot of stat(2)s, so a
 * pointer to a struct stat can be passed in.  "create" is
 * a bitmask, ored with the flags field in the overlay structure.
 * If it matches, the operation of the function is modified
 * for that overlay - it creates directories in the overlay
 * to make it possible to create "pathname"
 */

static char * nextstat1(char **pathnameptr, struct stat *statptr,
			struct overlay *overlay, int create, int *deletedp)
{
  char *pathname = *pathnameptr;
  char *new_pathname;
  char *p, *sp, *np;
  char buf[MAXPATHLEN];

  if (overlay == NULL
      || strncmp(pathname, overlay->mountpoint, overlay->mountpointlen) != 0
      || (pathname[overlay->mountpointlen] != '/'
	  && pathname[overlay->mountpointlen] != '\0')) {
    return NULL;
  }

  /* Construct the "new" pathname, with the mount point prefix replaced
   * with the real name; the tail end of pathname is preserved
   */

  /* XXX don't assume MAXPATHLEN, and throw the error return */

  if ((new_pathname = (char *) malloc(MAXPATHLEN)) == NULL) {
    errno = ENOMEM;
    return NULL;
  }

  strcpy(new_pathname, overlay->realname);

  p = pathname + overlay->mountpointlen;
  np = new_pathname + overlay->realnamelen;

  while (*p != '\0') {
    if ((sp = strchr(p, '/')) != NULL) {
      while (p < sp) *(np++) = *(p++);
    } else {
      while (*p) *(np++) = *(p++);
    }

    strcpy(np, "[DELETED]");
    DPRINTF((stderr, "nextstat1: lstating '%s'\n", new_pathname));
    if ((*nextlxstat)(_STAT_VER, new_pathname, statptr) == 0) {
      *deletedp = 1;
    }
    *np = '\0';

    DPRINTF((stderr, "nextstat1: lstating '%s'\n", new_pathname));
    if ((*nextlxstat)(_STAT_VER, new_pathname, statptr) == -1) {
      if (overlay->flags & create) {
	if (*p == '/') {
	  mkdir(new_pathname, 0777);
	} else {
	  return new_pathname;
	}
      } else {
	return NULL;
      }
    } else if (S_ISLNK(statptr->st_mode)) {
      if ((*nextreadlink)(new_pathname, buf, sizeof(buf)) == -1) {
	return NULL;
      }
      if (buf[0] == '!') {
	/* Rewrite path, change pathnamep, and goto top of this code */
      }
    }

    if (sp) {
      *(np++) = '/';
      while (*p == '/') p++;
    }
  }

  /* We do this for two reasons.  If the last item is a symlink, the
   * statptr will be set for the link, not the target.  Also, if
   * the function is called on the mount point itself, we need
   * to make sure it gets stat()ed (the above code won't run
   * at all in this case)
   */

#if 0
  DPRINTF((stderr, "nextstat1: stating '%s'\n", new_pathname));
  if ((*nextxstat)(_STAT_VER, new_pathname, statptr) == 0) {
    return new_pathname;
  } else {
    return NULL;
  }
#else
  DPRINTF((stderr, "nextstat1: stating '%s'\n", new_pathname));
  (*nextxstat)(_STAT_VER, new_pathname, statptr);
  return new_pathname;
#endif
}

static char * nextstat(const char *pathname, struct stat *statptr, int create)
{
  int deleted=0;
  char *new_pathname;
  struct overlay *overlay;

  for (overlay = overlays; overlay != NULL; overlay = overlay->next) {

    new_pathname = nextstat1(&pathname, statptr, overlay, create, &deleted);

    if ((new_pathname != NULL) || deleted) {
      DPRINTF((stderr, "nextstat: returning new '%s'\n", new_pathname));
      return new_pathname;
    }
  }

  if ((create == 0) && ((*nextlxstat)(_STAT_VER, pathname, statptr) == 0)) {
    DPRINTF((stderr, "nextstat: returning old '%s'\n", pathname));
    return pathname;
  } else {
    return NULL;
  }
}

static const char * overlays_match(char * result, const char * pathname,
				   enum mustexist mustexist)
{
  struct stat statstruct;
  char * realpathname;

  realpathname = nextstat(pathname, &statstruct, 0);

  /* In order for the unions.so module to work correctly, the cwd.so
   * module must be loaded before it, to ensure that all pathnames
   * passed down are absolute.  So why do we need this check?  Because
   * when cwd.so is initializing, it does a getcwd()
   *
   * Other places need to open a directory cleanly.  Programs need
   * to use opendir/readdir for the directory interface to work,
   * so open() should always return the actual directory
   */

  if ((realpathname != NULL) && S_ISDIR(statstruct.st_mode)) {
    DPRINTF((stderr, "overlays_match: directory return '%s'\n", pathname));
    return pathname;
  }

  switch (mustexist) {

  case NoExistanceCheck:
    return realpathname;

  /* If we're creating a file, first see if something's
   * already there.  If so, return EEXIST.  This code is the difference
   * between Create and CreateNoCheck.
   */

  case Create:
    if (realpathname != NULL) {
      errno = EEXIST;
      return NULL;
    }
    /* Fallthrough */

  case CreateNoCheck:
    return nextstat(pathname, &statstruct, CreateHere);

  case LinkMustExist:
    return realpathname;

  case DirMustExist:
    if ((realpathname != NULL) && S_ISDIR(statstruct.st_mode)) {
      return realpathname;
    } else {
      errno = ENOENT;
      return NULL;
    }

  case MustExist:
    /* XXX have to follow symlink here */
    return realpathname;
  }

  /* NOTREACHED */
  return NULL;
}

/* copy() - Utility function used by open() when it needs to perform
 * a copy-on-write.
 */

static void copy (const char *fromfile, const char *tofile)
{
  int fromfd, tofd;
  unsigned char buf[1024];
  size_t count;
  struct stat statstruct;

  if ((*nextxstat)(_STAT_VER, fromfile, &statstruct) == -1) {
    return;
  }
  if ((fromfd = open(fromfile, O_RDONLY)) == -1) {
    return;
  }

#define AllPerms (S_ISUID|S_ISGID|S_IRWXU|S_IRWXG|S_IRWXO)

  if ((tofd = open(tofile, O_CREAT | O_WRONLY,
		   statstruct.st_mode & AllPerms)) == -1) {
    close(fromfd);
    return;
  }
  while ((count = read(fromfd, buf, sizeof(buf))) > 0) {
    write(tofd, buf, count);
  }
  close(tofd);
  close(fromfd);
}

int open (const char *pathname, int flags, ...)
{
  char result[MAXPATHLEN];
  va_list ap;
  int mode;
  int exists;
  const char *new_pathname;
  struct stat statstruct;

  DPRINTF((stderr, "unions open: pathname is '%s'\n", pathname));

  if ((flags & O_ACCMODE) == O_RDONLY) {

    /* Read-only access (this is the easiest case) */

    DPRINTF((stderr, "unions open: O_RDONLY code\n"));

    if ((pathname = overlays_match(result, pathname, MustExist)) == NULL) {
      return -1;
    } else {
      DPRINTF((stderr, "unions open: O_RDONLY '%s'\n", pathname));
      return (*nextopen)(pathname, flags);
    }

  } else {

    /* Read-write, Write-only, and/or Create */

    DPRINTF((stderr, "unions open: WRITE code\n"));

    exists = (nextstat(pathname, &statstruct, 0) != NULL);

    if (exists) {

      /* Special case for devices - we allow them to be opened writable
       * even if an overlay was flagged CopyOnWrite
       */

      if (S_ISCHR(statstruct.st_mode) || S_ISBLK(statstruct.st_mode)) {
	DPRINTF((stderr, "unions open: write access, `%s' is device\n",
		 pathname));
	return (*nextopen)(pathname, flags);
      }

      /* Bail if file exists and user specified exclusive create */

      if ((flags & O_CREAT) && (flags & O_EXCL)) {
	errno = EEXIST;
	return -1;
      }

      new_pathname = nextstat(pathname, &statstruct, CopyOnWrite);

      if (new_pathname == NULL) {
	new_pathname = pathname;
      } else {
	copy(pathname, new_pathname);
      }

      DPRINTF((stderr, "unions open: write access, `%s' copyonwrite\n",
	       new_pathname));

      return (*nextopen)(new_pathname, flags);

    } else {

      /* Bail if file doesn't exist and user didn't specify create */

      if ( !(flags & O_CREAT)) {
	errno = ENOENT;
	return -1;
      }

      va_start(ap, flags);
      mode = va_arg(ap, int);
      va_end(ap);

      new_pathname = nextstat(pathname, &statstruct, CreateHere);

      if (new_pathname == NULL) new_pathname = pathname;

      DPRINTF((stderr, "unions open: write access, `%s' create\n",
	       new_pathname));

      return (*nextopen)(new_pathname, flags, mode);
    }
  }
}

__ALIAS(int, open, (const char *, int flags, ...));


static int (*nextunlink) (const char *) = NULL;
int unlink(const char *path)
{
  const char *errval;
  char result[MAXPATHLEN];
  if (nextunlink == NULL) {
    nextunlink = dlsym(RTLD_NEXT, "unlink");
    if ((errval = dlerror()) != NULL) {
      fprintf(stderr, "dlsym(unlink): %s\n", errval);
    }
  }
  DPRINTF((stderr, "unions: Entering unlink; path='%s'\n", path));

  /* This code really needs to be a lot more precise than this.
   *
   * We need a new overlay option called "delete".  Any overlay flagged
   * with this option will intercept unlink() calls and cause the
   * file to be deleted to be noted in a special file rather than
   * actually being deleted.
   *
   * Similar actions for rmdir() and rename()
   */

  if ((path = overlays_match(result, path, LinkMustExist)) == NULL) {
    return -1;
  }
  return (*nextunlink) (path);
}
__ALIAS(int, unlink, (const char *));

/* The remaining functions that need to be overridden all follow a
 * standard format.  The only thing we need to do is toss in one
 * or two calls to overlays_match() before passing the results on to
 * the original function.  Of course, the "original function" needs
 * to be found via a dlsym(3) lookup.  These macros makes things
 * a bit simplier, at some expense in readability...
 */

#define FETCH_DLSYM(syscall)						\
  if (next##syscall == NULL) {						\
    const char *errval;							\
    next##syscall = dlsym(RTLD_NEXT, #syscall);				\
    if (next##syscall == NULL ||					\
	(errval = dlerror()) != NULL) {					\
      next##syscall = dlsym(RTLD_NEXT, "__" #syscall);			\
    }									\
    if (next##syscall == NULL ||					\
	(errval = dlerror()) != NULL) {					\
      fprintf(stderr, "dlsym(" #syscall "): %s\n", errval);		\
    }									\
  }									\

#define OVERRIDE_SYSCALL(type, syscall, arglist, calllist, var, cond)	\
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  char result[MAXPATHLEN];						\
  FETCH_DLSYM(syscall);							\
  DPRINTF((stderr, "unions: Entering " #syscall "; path='%s'\n", path));\
  if ((var = overlays_match(result, var, cond)) == NULL)		\
    return -1;								\
  return (*next##syscall) calllist;					\
}									\
__ALIAS(type, syscall, arglist);

#define OVERRIDE_SYSCALL2(type, syscall, arglist, calllist, var1, cond1, var2,cond2)	\
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  char result1[MAXPATHLEN];						\
  char result2[MAXPATHLEN];						\
  FETCH_DLSYM(syscall);							\
  DPRINTF((stderr, "unions: Entering " #syscall "; path='%s'\n", path));\
  if ((var1 = overlays_match(result1, var1, cond1)) == NULL)		\
    return -1;								\
  if ((var2 = overlays_match(result2, var2, cond2)) == NULL)		\
    return -1;								\
  return (*next##syscall) calllist;					\
}									\
__ALIAS(type, syscall, arglist);

OVERRIDE_SYSCALL(int, execve,
		 (const char *path, char *const argv[], char *const envp[]),
		 (path, argv, envp), path, MustExist);

OVERRIDE_SYSCALL(int, xstat, (int ver, const char *path, struct stat *buf),
		 (ver, path, buf), path, MustExist);
OVERRIDE_SYSCALL(int, lxstat, (int ver, const char *path, struct stat *buf),
		 (ver, path, buf), path, LinkMustExist);
OVERRIDE_SYSCALL(int, xstat64, (int ver, const char *path, struct stat64 *buf),
		 (ver, path, buf), path, MustExist);
OVERRIDE_SYSCALL(int, lxstat64, (int ver, const char *path, struct stat64 *buf),
		 (ver, path, buf), path, LinkMustExist);
OVERRIDE_SYSCALL(int, xmknod,
		 (int ver, const char *path, mode_t mode, dev_t *dev),
		 (ver, path, mode, dev), path, Create);

OVERRIDE_SYSCALL(int, access, (const char *path, int mode),
		 (path, mode), path, MustExist);
OVERRIDE_SYSCALL(int, acct, (const char *path), (path), path, MustExist);
OVERRIDE_SYSCALL(int, chmod, (const char *path, mode_t mode),
		 (path, mode), path, MustExist);
OVERRIDE_SYSCALL(int, chown, (const char *path, uid_t uid, gid_t gid),
		 (path, uid, gid), path, MustExist);
OVERRIDE_SYSCALL2(int, link, (const char *path, const char *newpath),
		  (path, newpath), path, LinkMustExist, newpath, Create);
OVERRIDE_SYSCALL(int, mkdir, (const char *path, mode_t mode),
		 (path, mode), path, Create);
OVERRIDE_SYSCALL(int, symlink, (const char *path, const char *newpath),
		 (path, newpath), newpath, Create);
OVERRIDE_SYSCALL2(int, rename, (const char *path, const char *newpath),
		  (path, newpath), path, LinkMustExist, newpath, CreateNoCheck);
OVERRIDE_SYSCALL(int, readlink, (const char *path, char *buf, size_t bufsiz),
		 (path, buf, bufsiz), path, LinkMustExist);
OVERRIDE_SYSCALL(int, rmdir, (const char *path), (path), path, LinkMustExist);
OVERRIDE_SYSCALL(int, utime, (const char *path, const struct utimbuf *buf),
		 (path, buf), path, MustExist);

/* opendir/readdir strategy
 *
 * Upon calling opendir(), we read each of the overlayed directories
 * into memory, in their entirety, and build up a single list of
 * struct dirent's.  If a name is duplicated, the lower level entry is
 * flagged deleted (only in the in-memory copy), so the higher level
 * overlay takes precedence.  readdir() then returns pointers to these
 * structures, without any further low level reads.
 *
 * Non-overlayed directories are a special case, noted by a flag (usenextDIR)
 * in the "myDIR" structures (a substitute our functions return in place
 * of the expected DIR pointer).
 */

struct mydirstream {
  int usetrueDIR;		/* If true, use trueDIR with original funcs */
  DIR *trueDIR;

  char *data;			/* Directory block.  */
  size_t allocation;		/* Space allocated for the block.  */
  size_t size;			/* Total valid data in the block.  */
  size_t offset;		/* Current offset into the block.  */

  /* Mutex lock needed for this structure.  */
};

typedef struct mydirstream myDIR;

#ifdef _DIRENT_HAVE_D_RECLEN
#define RECLEN(dp) (dp->d_reclen)
#else
      /* The only version of `struct dirent' that lacks `d_reclen'
	 is fixed-size.  */
#define RECLEN(dp) (sizeof *dp);
#endif

/* overlay_directory() - helper function called from opendir()
 *
 * Read all the entries from directory "name", and add those
 * that don't already exist to "dirp".
 * It's OK for the directory not to exist - we just do nothing.
 * NULL return on error.
 */

static myDIR * overlay_directory (myDIR *dirp, const char *name)
{
  DIR *thisdirp;
  struct dirent64 *entryp;

  DPRINTF((stderr, "Entering overlay_directory(0x%x, '%s')\n",
	   (int)dirp, name));

  if ((thisdirp = (*nextopendir)(name))) {
    while ((entryp = (*nextreaddir64)(thisdirp))) {
      int i;

      /* Got a directory entry.  If it's the same as one we've already
       * seen, skip it
       */

      if (dirp->size > 0) {
	struct dirent64 *preventryp;

	for (i = 0; i < dirp->size; i += RECLEN(preventryp)) {
	  preventryp = (struct dirent64 *) (dirp->data + i);
	  if (alphasort64(&entryp, &preventryp) == 0) {
	    DPRINTF((stderr, "Directory entries match: `%s', `%s'\n",
		     entryp->d_name, preventryp->d_name));
	    break;
	  }
	}
      }

      /* OK, we haven't seen the name before.  Add it to the list,
       * enlarging the data block to make room, if needed.
       */

      if (dirp->size == 0 || i == dirp->size) {
	if (RECLEN(entryp) > (dirp->allocation - dirp->size)) {
	  if (dirp->allocation == 0) {
	    dirp->allocation = 10 * sizeof(*entryp);
	  } else {
	    dirp->allocation <<= 2;
	  }
	  dirp->data = realloc(dirp->data, dirp->allocation);
	  if (dirp->data == NULL) {
	    return NULL;
	  }
	}
	memcpy(dirp->data + dirp->size, entryp, RECLEN(entryp));
	dirp->size += RECLEN(entryp);
      }
    }
  }
  return dirp;
}

DIR * opendir (const char *name)
{
  char *new_pathname=NULL;
  myDIR *dirp=NULL;
  struct overlay *overlay;
  struct stat statstruct;
  int deleted=0;

  DPRINTF((stderr, "Entering __opendir('%s')\n", name));

  if ((dirp = malloc(sizeof(*dirp))) == NULL) {
    goto abort;
  }

  dirp->usetrueDIR = 1;
  dirp->data = NULL;
  dirp->allocation = 0;
  dirp->size = 0;
  dirp->offset = 0;

  for (overlay = overlays; overlay; overlay = overlay->next) {
    new_pathname = nextstat1(&name, &statstruct, overlay, 0, &deleted);
    if ((new_pathname != NULL) && S_ISDIR(statstruct.st_mode)) {
      if (overlay_directory(dirp, new_pathname) == NULL) {
	goto abort;
      }
      dirp->usetrueDIR = 0;
    }
    /* XXX not quite right, since we need to prevent the original dir
     * from being overlay, too
     */
    if (deleted) break;
  }

  if (dirp->usetrueDIR) {
    /* If we get to this point with usetrueDIR still true, then
     * none of the overlays matched this dir
     */
    if ((dirp->trueDIR = (*nextopendir)(name)) == NULL) {
      goto abort;
    }
  } else {
    /* Otherwise, the original directory (if it exists) is the last
     * one to add to our overlay
     */
    if (overlay_directory(dirp, name) == NULL) {
      goto abort;
    }
  }

  if (dirp->trueDIR) {
    DPRINTF((stderr, "Leaving __opendir; using trueDIR\n"));
  } else {
    DPRINTF((stderr, "Leaving __opendir; emulating\n"));
  }

  return (DIR *) dirp;

 abort:
  if (dirp && dirp->data) {
    free (dirp->data);
  }
  if (dirp) {
    free(dirp);
  }
  if (new_pathname) {
    free(new_pathname);
  }
  DPRINTF((stderr, "Leaving __opendir via abort\n"));
  return NULL;
}

__ALIAS(DIR *, opendir, (const char *));

/* This code doesn't strictly confirm to POSIX, since it returns a pointer
 * to a structure that gets reused instead of discarded.
 */

struct dirent64 *readdir64(DIR *arg)
{
  myDIR *dirp = (myDIR *)arg;
  struct dirent64 *dp = NULL;

  DPRINTF((stderr, "unions: Entering __readdir\n"));

  if (dirp->usetrueDIR) {
    dp = (*nextreaddir64)(dirp->trueDIR);
  } else {

    /* Lock structure dirp */

    if (dirp->offset < dirp->size) {

      dp = (struct dirent64 *) &dirp->data[dirp->offset];

      dirp->offset += RECLEN(dp);
    }

    /* Unlock structure dirp */
  }
  return dp;
}

__ALIAS(struct dirent *, readdir64, (DIR *));

struct dirent *readdir(DIR *arg)
{
  myDIR *dirp = (myDIR *)arg;
  struct dirent64 *dp = NULL;
  static struct dirent retstruct;

  DPRINTF((stderr, "unions: Entering __readdir\n"));

  if (dirp->usetrueDIR) {
    return (*nextreaddir)(dirp->trueDIR);
  } else {

    /* Lock structure dirp */

    if (dirp->offset < dirp->size) {

      dp = (struct dirent64 *) &dirp->data[dirp->offset];

      dirp->offset += RECLEN(dp);
    }

    /* Unlock structure dirp */
  }

  if (dp) {
    retstruct.d_ino = dp->d_ino;
    retstruct.d_off = dp->d_off;
    retstruct.d_reclen = dp->d_reclen;
    retstruct.d_type = dp->d_type;
    strcpy(retstruct.d_name, dp->d_name);
    return &retstruct;
  } else {
    return NULL;
  }
}

__ALIAS(struct dirent *, readdir, (DIR *));

int closedir(DIR *arg)
{
  myDIR *dirp = (myDIR *)arg;
  int retval = 0;

  if (dirp->usetrueDIR) {
    retval = (*nextclosedir)(dirp->trueDIR);
  }

  if (dirp->data) {
    free(dirp->data);
  }
  free(dirp);
  return retval;
}

__ALIAS(int, closedir, (DIR *));

void rewinddir(DIR * arg)
{
  myDIR *dirp = (myDIR *)arg;

  if (dirp->usetrueDIR) {
    (*nextrewinddir)(dirp->trueDIR);
  } else {
    /* Lock structure dirp */
    dirp->offset = 0;
    /* Unlock structure dirp */
  }
}

__ALIAS(void, rewinddir, (DIR *));

off_t telldir(DIR * arg)
{
  myDIR *dirp = (myDIR *)arg;

  if (dirp->usetrueDIR) {
    return (*nexttelldir)(dirp->trueDIR);
  } else {
    /* Lock structure dirp */
    return dirp->offset;
    /* Unlock structure dirp */
  }
}

__ALIAS(off_t, telldir, (DIR *));

void seekdir(DIR * arg, off_t offset)
{
  myDIR *dirp = (myDIR *)arg;

  if (dirp->usetrueDIR) {
    (*nextseekdir)(dirp->trueDIR, offset);
  } else {
    /* Lock structure dirp */
    dirp->offset = offset;
    /* Unlock structure dirp */
  }
}

__ALIAS(void, seekdir, (DIR *, off_t));


/* Parse and build overlay structure for the string passed in as "mountcmd".
 *
 * NOTE: Writes NULs into mountcmd and stores pointers to it
 *
 * Grammer:
 * command -> opt_ws overlay opt_ws '@' opt_ws mountpoint opt_ws options
 * options -> '(' opt_ws optionlist ')'
 * options -> nil
 * optionlist -> option opt_ws ',' opt_ws optionlist
 * optionlist -> nil
 */

static void parse_mount(char *mountcmd)
{
  char *realname;
  char *mountpoint;
  struct stat statstruct;
  struct overlay *overlay;
  int flags = 0;

  while (isspace(*mountcmd)) mountcmd ++;

  realname = mountcmd;

  for (; *mountcmd != '\0' && *mountcmd != '@'; mountcmd++);

  if (mountcmd == '\0') {
    fprintf(stderr, "Mount syntax: overlay@mountpoint(options)\n");
    return;
  }
  *(mountcmd++) = '\0';

  if ((*nextxstat)(_STAT_VER, realname, &statstruct) == -1
      || !S_ISDIR(statstruct.st_mode)) {
    fprintf(stderr, "Overlay `%s' doesn't exist\n", realname);
    return;
  }

  while (isspace(*mountcmd)) mountcmd ++;

  mountpoint = mountcmd;

  for (; *mountcmd != '\0' && *mountcmd != '('; mountcmd++);

  if (*mountcmd == '(') {

    *(mountcmd++) = '\0';

    while (isspace(*mountcmd)) mountcmd ++;

    while (*mountcmd != ')') {
      if (strncasecmp(mountcmd, "create", 6) == 0) {
	flags |= CreateHere;
	mountcmd += 6;
      } else if (strncasecmp(mountcmd, "copyonwrite", 11) == 0) {
	flags |= CopyOnWrite;
	mountcmd += 11;
      } else if (*mountcmd == ',') {
	mountcmd ++;
      } else {
	fprintf(stderr, "Mount syntax: overlay@mountpoint(options)\n");
	return;
      }
      while (isspace(*mountcmd)) mountcmd ++;
    }

  }

  if ((*nextxstat)(_STAT_VER, mountpoint, &statstruct) == -1
      || !S_ISDIR(statstruct.st_mode)) {
    fprintf(stderr, "Mount point `%s' doesn't exist\n", mountpoint);
    return;
  }

  if ((overlay = (struct overlay *) malloc(sizeof(struct overlay))) == NULL) {
    fprintf(stderr, "No memory for struct overlay\n");
    return;
  }

  overlay->next = overlays;
  overlay->realname = realname;
  overlay->realnamelen = strlen(realname);
  overlay->mountpoint = mountpoint;
  overlay->mountpointlen = strlen(mountpoint);
  overlay->flags = flags;

  if (mountpoint[overlay->mountpointlen-1] == '/') {
    mountpoint[overlay->mountpointlen-1] = '\0';
    overlay->mountpointlen --;
  }

  DPRINTF((stderr, "New overlay: %s@%s %s %s\n", realname, mountpoint,
	   (flags & CopyOnWrite) ? "CopyOnWrite" : "NoCopyOnWrite",
	   (flags & CreateHere) ? "Create" : "NoCreate"));

  overlays = overlay;
}


static void parse_mounts(char *mountcmd)
{
  char *semicolon;

  while ((semicolon = index(mountcmd, ';'))) {
    *semicolon = '\0';
    parse_mount(mountcmd);
    mountcmd = semicolon+1;
  }
  parse_mount(mountcmd);
}


void _init(void)
{
  const char *errval;
  char *unions;

  DPRINTF((stderr, "unions: GotInit\n"));

  nextopen = dlsym(RTLD_NEXT,"open");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(open): %s\n", errval);
  }

  nextopendir = dlsym(RTLD_NEXT,"opendir");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(opendir): %s\n", errval);
  }

  nextclosedir = dlsym(RTLD_NEXT,"closedir");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(closedir): %s\n", errval);
  }

  nextreaddir = dlsym(RTLD_NEXT,"readdir");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(readdir): %s\n", errval);
  }

  nextreaddir64 = dlsym(RTLD_NEXT,"readdir64");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(readdir64): %s\n", errval);
  }

  nextreadlink = dlsym(RTLD_NEXT,"readlink");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(readlink): %s\n", errval);
  }

  nextmkdir = dlsym(RTLD_NEXT,"mkdir");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(mkdir): %s\n", errval);
  }

  /* These three functions (xstat, lxstat, xmknod) are special because
   * they only exist in "__" form.  The user-visible versions are
   * actually inline functions in the include files.
   */

  nextxstat = dlsym(RTLD_NEXT,"__xstat");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(__xstat): %s\n", errval);
  }

  nextlxstat = dlsym(RTLD_NEXT,"__lxstat");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(__lxstat): %s\n", errval);
  }

  nextxmknod = dlsym(RTLD_NEXT,"__xmknod");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(__xmknod): %s\n", errval);
  }

  if ((unions = getenv("UNIONS")) != NULL) {
    parse_mounts(strdup(unions));
  }

  DPRINTF((stderr, "unions: EndInit\n"));
}

void error ();

/* Ensure that the directory ARGPATH exists.
   Remove any trailing slashes from ARGPATH before calling this function.

   Create any leading directories that don't already exist, with
   permissions PARENT_MODE.
   If the last element of ARGPATH does not exist, create it as
   a new directory with permissions MODE.
   If OWNER and GROUP are non-negative, use them to set the UID and GID of
   any created directories.
   If VERBOSE_FMT_STRING is nonzero, use it as a printf format
   string for printing a message after successfully making a directory,
   with the name of the directory that was just made as an argument.
   If PRESERVE_EXISTING is non-zero and ARGPATH is an existing directory,
   then do not attempt to set its permissions and ownership.

   Return 0 if ARGPATH exists as a directory with the proper
   ownership and permissions when done, otherwise 1.  */

#if __STDC__
int
make_path (const char *argpath,
	   int mode,
	   int parent_mode,
	   uid_t owner,
	   gid_t group,
	   int preserve_existing,
	   const char *verbose_fmt_string)
#else
int
make_path (argpath, mode, parent_mode, owner, group, preserve_existing,
	   verbose_fmt_string)
     const char *argpath;
     int mode;
     int parent_mode;
     uid_t owner;
     gid_t group;
     int preserve_existing;
     const char *verbose_fmt_string;
#endif
{
  char *dirpath;		/* A copy we can scribble NULs on.  */
  struct stat stats;
  int retval = 0;
  int oldmask = umask (0);

  /* FIXME: move this alloca and strcpy into the if-block.
     Set dirpath to argpath in the else-block.  */
  dirpath = (char *) alloca (strlen (argpath) + 1);
  strcpy (dirpath, argpath);

  if (stat (dirpath, &stats))
    {
      char *slash;
      int tmp_mode;		/* Initial perms for leading dirs.  */
      int re_protect;		/* Should leading dirs be unwritable? */
      struct ptr_list
      {
	char *dirname_end;
	struct ptr_list *next;
      };
      struct ptr_list *p, *leading_dirs = NULL;

      /* If leading directories shouldn't be writable or executable,
	 or should have set[ug]id or sticky bits set and we are setting
	 their owners, we need to fix their permissions after making them.  */
      if (((parent_mode & 0300) != 0300)
	  || (owner != (uid_t) -1 && group != (gid_t) -1
	      && (parent_mode & 07000) != 0))
	{
	  tmp_mode = 0700;
	  re_protect = 1;
	}
      else
	{
	  tmp_mode = parent_mode;
	  re_protect = 0;
	}

      slash = dirpath;
      while (*slash == '/')
	slash++;
      while ((slash = strchr (slash, '/')))
	{
	  *slash = '\0';
	  if (stat (dirpath, &stats))
	    {
	      if (nextmkdir (dirpath, tmp_mode))
		{
		  error (0, errno, "cannot create directory `%s'", dirpath);
		  umask (oldmask);
		  return 1;
		}
	      else
		{
		  if (verbose_fmt_string != NULL)
		    error (0, 0, verbose_fmt_string, dirpath);

		  if (owner != (uid_t) -1 && group != (gid_t) -1
		      && chown (dirpath, owner, group)
#if defined(AFS) && defined (EPERM)
		      && errno != EPERM
#endif
		      )
		    {
		      error (0, errno, "%s", dirpath);
		      retval = 1;
		    }
		  if (re_protect)
		    {
		      struct ptr_list *new = (struct ptr_list *)
			alloca (sizeof (struct ptr_list));
		      new->dirname_end = slash;
		      new->next = leading_dirs;
		      leading_dirs = new;
		    }
		}
	    }
	  else if (!S_ISDIR (stats.st_mode))
	    {
	      error (0, 0, "`%s' exists but is not a directory", dirpath);
	      umask (oldmask);
	      return 1;
	    }

	  *slash++ = '/';

	  /* Avoid unnecessary calls to `stat' when given
	     pathnames containing multiple adjacent slashes.  */
	  while (*slash == '/')
	    slash++;
	}

      /* We're done making leading directories.
	 Create the final component of the path.  */

      /* The path could end in "/." or contain "/..", so test
	 if we really have to create the directory.  */

      if (nextxstat (_STAT_VER, dirpath, &stats) && nextmkdir (dirpath, mode))
	{
	  error (0, errno, "cannot create directory `%s'", dirpath);
	  umask (oldmask);
	  return 1;
	}
      if (verbose_fmt_string != NULL)
	error (0, 0, verbose_fmt_string, dirpath);

      if (owner != (uid_t) -1 && group != (gid_t) -1)
	{
	  if (chown (dirpath, owner, group)
#ifdef AFS
	      && errno != EPERM
#endif
	      )
	    {
	      error (0, errno, "%s", dirpath);
	      retval = 1;
	    }
	  /* chown may have turned off some permission bits we wanted.  */
	  if ((mode & 07000) != 0 && chmod (dirpath, mode))
	    {
	      error (0, errno, "%s", dirpath);
	      retval = 1;
	    }
	}

      /* If the mode for leading directories didn't include owner "wx"
	 privileges, we have to reset their protections to the correct
	 value.  */
      for (p = leading_dirs; p != NULL; p = p->next)
	{
	  *(p->dirname_end) = '\0';
	  if (chmod (dirpath, parent_mode))
	    {
	      error (0, errno, "%s", dirpath);
	      retval = 1;
	    }
	}
    }
  else
    {
      /* We get here if the entire path already exists.  */

      if (!S_ISDIR (stats.st_mode))
	{
	  error (0, 0, "`%s' exists but is not a directory", dirpath);
	  umask (oldmask);
	  return 1;
	}

      if (!preserve_existing)
	{
	  /* chown must precede chmod because on some systems,
	     chown clears the set[ug]id bits for non-superusers,
	     resulting in incorrect permissions.
	     On System V, users can give away files with chown and then not
	     be able to chmod them.  So don't give files away.  */

	  if (owner != (uid_t) -1 && group != (gid_t) -1
	      && chown (dirpath, owner, group)
#ifdef AFS
	      && errno != EPERM
#endif
	      )
	    {
	      error (0, errno, "%s", dirpath);
	      retval = 1;
	    }
	  if (chmod (dirpath, mode))
	    {
	      error (0, errno, "%s", dirpath);
	      retval = 1;
	    }
	}
    }

  umask (oldmask);
  return retval;
}
