
#define _GNU_SOURCE

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dlfcn.h>

/* TRACE module
 *
 * Preload before libc to trace system calls.
 *
 * Compile with:
 *    gcc -nostartfiles -fpic -shared -o trace.so trace.c -ldl
 *
 * Usage:
 *    export LD_PRELOAD=/usr/local/lib/trace.so
 *
 */



static int debuglevel = 0;
FILE * debugfile = NULL;

static inline void debug(int level, const char *format, ...) {
  va_list ap;

  va_start(ap, format);
  if (level <= debuglevel && debugfile) {
    fprintf(debugfile, "%d:", getpid());
    vfprintf(debugfile, format, ap);
    fflush(debugfile);
  }
  va_end(ap);
}


/* The remaining functions that need to be overridden all follow a
 * standard format.  The only thing we need to do is toss in one
 * or two calls to resolve_path() before passing the results on to
 * the original function.  The macro makes things a bit simplier,
 * at some expense in readability...
 */

#define FETCH_DLSYM(syscall)						\
  if (next##syscall == NULL) {						\
    const char *errval;							\
    next##syscall = dlsym(RTLD_NEXT, #syscall);				\
    if (next##syscall == NULL ||					\
	(errval = dlerror()) != NULL) {					\
      next##syscall = dlsym(RTLD_NEXT, "__" #syscall);			\
    }									\
    if (next##syscall == NULL ||					\
	(errval = dlerror()) != NULL) {					\
      fprintf(stderr, "dlsym(" #syscall "): %s\n", errval);		\
    }									\
  }

#define __ALIAS(type, syscall, arglist)					\
   type __##syscall arglist __attribute__ ((alias( #syscall )))


static int (*nextopen)(const char *, int, ...) = NULL;
int open (const char *path, int flags, ...)
{
  va_list ap;
  int mode;
  int retval;

  va_start(ap, flags);
  mode = va_arg(ap, int);
  va_end(ap);

  FETCH_DLSYM(open);
  debug(1, "open(%s)", path);
  retval = (*nextopen)(path, flags, mode);
  debug(1, " = %d\n", retval);
  return retval;
}
__ALIAS(int, open, (const char *path, int flags, ...));


#define OVERRIDE_SYSCALL(type, syscall, arglist, varlist, format, var)	\
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  type retval;								\
  FETCH_DLSYM(syscall);							\
  debug(1, format, var);						\
  retval = (*next##syscall) varlist;					\
  debug(1, " = %d\n", retval);						\
  return retval;							\
}									\
__ALIAS(type, syscall, arglist)

#define OVERRIDE_SYSCALL2(type, syscall, arglist, varlist, format, var1, var2)\
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  type retval;								\
  FETCH_DLSYM(syscall);							\
  debug(1, format, var1, var2);						\
  retval = (*next##syscall) varlist;					\
  debug(1, " = %d\n", retval);						\
  return retval;							\
}									\
__ALIAS(type, syscall, arglist)

#define OVERRIDE_SYSCALL3(type, syscall, arglist, varlist, format, var1, var2, var3)\
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  type retval;								\
  FETCH_DLSYM(syscall);							\
  debug(1, format, var1, var2, var3);					\
  retval = (*next##syscall) varlist;					\
  debug(1, " = %d\n", retval);						\
  return retval;							\
}									\
__ALIAS(type, syscall, arglist)

#define OVERRIDE_SYSCALLb(type, syscall, arglist, varlist, format, var, buf)\
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  type retval;								\
  FETCH_DLSYM(syscall);							\
  debug(1, format, var);						\
  retval = (*next##syscall) varlist;					\
  if (retval > 0) debug(1, " = %d(%.*s)\n", retval, retval, buf);	\
  else debug(1, " = %d\n", retval);					\
  return retval;							\
}									\
__ALIAS(type, syscall, arglist)

#define OVERRIDE_SYSCALLd(type, syscall, arglist, varlist, format, var)	\
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  type retval;								\
  FETCH_DLSYM(syscall);							\
  debug(1, format, var);						\
  retval = (*next##syscall) varlist;					\
  debug(1, "\n");							\
  return retval;							\
}									\
__ALIAS(type, syscall, arglist)


OVERRIDE_SYSCALL(int, xstat, (int ver, const char *path, struct stat *buf), (ver, path, buf), "xstat(%s)", path);
OVERRIDE_SYSCALL(int, lxstat, (int ver, const char *path, struct stat *buf), (ver, path, buf), "lxstat(%s)", path);
OVERRIDE_SYSCALL(int, xstat64, (int ver, const char *path, struct stat64 *buf), (ver, path, buf), "xstat64(%s)", path);
OVERRIDE_SYSCALL(int, lxstat64, (int ver, const char *path, struct stat64 *buf), (ver, path, buf), "lxstat64(%s)", path);
OVERRIDE_SYSCALL2(int, xmknod, (int ver, const char *path, mode_t mode, dev_t *dev), (ver, path, mode, dev), "xmknod(%s,%04o)", path, mode);

OVERRIDE_SYSCALL2(int, access, (const char *path, int mode), (path, mode), "access(%s,%d)", path, mode);
OVERRIDE_SYSCALL(int, acct, (const char *path), (path), "acct(%s)", path);

OVERRIDE_SYSCALL2(int, chmod, (const char *path, mode_t mode), (path, mode), "chmod(%s,%04o)", path, mode);
OVERRIDE_SYSCALL3(int, chown, (const char *path, uid_t uid, gid_t gid), (path, uid, gid), "chown(%s,%d,%d)", path, uid, gid);

OVERRIDE_SYSCALL2(int, link, (const char *path, const char *path2), (path, path2), "link(%s,%s)", path, path2);

OVERRIDE_SYSCALL(int, unlink, (const char *path), (path), "unlink(%s)", path);
OVERRIDE_SYSCALLb(int, readlink, (const char *path, char *buf, size_t bufsiz), (path, buf, bufsiz), "readlink(%s)", path, buf);

OVERRIDE_SYSCALL2(int, symlink, (const char *symlink, const char *path), (symlink, path), "symlink(%s,%s)", symlink, path);

OVERRIDE_SYSCALL2(int, rename, (const char *path, const char *path2), (path, path2), "rename(%s,%s)", path, path2);

OVERRIDE_SYSCALL2(int, mkdir, (const char *path, mode_t mode), (path, mode), "mkdir(%s,%04o)", path, mode);
OVERRIDE_SYSCALL(int, rmdir, (const char *path), (path), "rmdir(%s)", path);

OVERRIDE_SYSCALL(int, utime, (const char *path, const struct utimbuf *buf), (path, buf), "utime(%s)", path);

OVERRIDE_SYSCALLd(DIR *, opendir, (const char *path), (path), "opendir(%s)", path);

void _init(void)
{
  char *envvar;

  if ((envvar = getenv("TRACELEVEL")) != NULL) {
    debuglevel = atoi(envvar);
  }

  if ((envvar = getenv("DEBUGFILE")) != NULL) {
    debugfile = fopen(envvar, "a");
  }

  if (debugfile == NULL) {
    debugfile = stderr;
  }
}
