
#define _GNU_SOURCE

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <dlfcn.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include <semaphore.h>

/* CWD module
 *
 * Preload before libc to maintain cwd in user space and convert all
 * pathnames used in system calls into absolute paths.  Major use is
 * before virtual filesystem modules that make new directories appear.
 * Unfortunately, since we might chdir to a directory that doesn't
 * actually exist, we can't use the chdir() or fchdir() system calls.
 * Instead, we need this module to intercept and mimic them, along
 * with anything else that might depend on cwd.
 *
 *
 * Compile with:
 *    gcc -nostartfiles -fpic -shared -o cwd.so cwd.c -ldl
 *
 * System-wide usage:
 *    /etc/ld.so.preload should contain the line:
 *	/usr/local/lib/cwd.so
 *
 * User or process specific usage:
 *    export LD_PRELOAD=/usr/local/lib/cwd.so
 *	(only needed if cwd.so doesn't appear in /etc/ld.so.preload)
 *
 */

#define DEBUG 0

/* #define __OVERRIDE 1 to generate aliases for __ varients of 
 * all overridden functions
 */

#define __OVERRIDE 1

#define OVERRIDE_getcwd 1



#if DEBUG
#define DPRINTF(args) fprintf args
#else
#define DPRINTF(args)
#endif

#if __OVERRIDE
#define __ALIAS(type, syscall, arglist) \
   type __##syscall arglist __attribute__ ((alias( #syscall )))
#else
#define __ALIAS(type, syscall, arglist)
#endif



static char cwd[MAXPATHLEN];

static int enable_module = 0;

/* Table of functions we override
 *
 * FUNCTION NAME       USE ORIGINAL             WHY
 *                     RECURSIVELY?
 *
 * chdir                   no             cwd kept in user space
 * fchdir                  no             cwd kept in user space
 * open                    yes            depends on cwd
 * xstat                   yes            depends on cwd
 * lstat                   yes            depends on cwd
 * execve                  yes            depends on cwd
 * access                  yes            depends on cwd
 * acct                    yes            depends on cwd
 * chmod                   yes            depends on cwd
 * chown                   yes            depends on cwd
 * link                    yes            depends on cwd
 * unlink                  yes            depends on cwd
 * symlink                 yes            depends on cwd
 * readlink                yes            depends on cwd
 * rename                  yes            depends on cwd
 * mkdir                   yes            depends on cwd
 * rmdir                   yes            depends on cwd
 * mknod                   yes            depends on cwd
 * opendir                 yes            depends on cwd
 *                                          POSIX directory function
 *                                          easier than overriding getdents()
 * getcwd                  no             efficiency
 * utime                   yes            depends on cwd
 *
 * TO BE DONE:
 *
 * chroot                  yes            how???
 * bind                    yes            depends (UNIX domain) on cwd
 * connect                 yes            depends (UNIX domain) on cwd
 */

enum symlink_flag { follow_symlinks, show_last_symlink };

/* expand_path() is a helper function used by resolve_path()
 * It adds "pathname" to "result", the end of which is tracked by "p"
 * It recurses if it needs to follow a symlink, and returns the new "p"
 * WARNING: Doesn't check for buffer overflows :-(
 */

static char * expand_path (const char *pathname,
			   enum symlink_flag symlink_flag,
			   char *result, char *p, int symlink_count)
{
  if (pathname[0] == '/') {

    /* If the pathname is absolute, discard everything we done so far
     * by reseting "p" to point to the beginning of the result buffer
     */

    p=result;
  }

  while (*pathname) {

    if (pathname[0] == '/') {

      /* Directory seperator - copy it to the output */

      *(p++) = '/';
      pathname ++;

    } else if (pathname[0] == '.'
	       && (pathname[1] == '/' || pathname[1] == '\0')) {

      /* Current directory "." - just skip it */

      pathname ++;

    } else if (pathname[0] == '.' && pathname[1] == '.'
	       && (pathname[2] == '/' || pathname[2] == '\0')) {

      /* Parent directory ".." - backup at least one char (the slash
       * that must be at the end of result buffer), then throw away
       * an entire component by skipping back to the last slash
       */

      p --;
      if (*p != '/') DPRINTF((stderr, "No slash prior to ..!\n"));
      do {
	if (p > result) p --;
      } while (p > result && (*p) != '/');

      pathname += 2;

    } else {

      /* Otherwise, normal entry.  Copy to the output buffer. */

      struct stat statstruct;
      char *savedp = p;

      while ((*pathname != '\0') && (*pathname != '/')) {
	*(p++) = *(pathname++);
      }

      /* Check to see if we're looking at a symbolic link.
       * If so, follow it, unless we're at the end of the pathname
       * and are flagged to show the last symlink
       */

      if (*pathname != '\0' || symlink_flag != show_last_symlink) {

	*p = '\0';

	if (lstat(result, &statstruct) == 0 && S_ISLNK(statstruct.st_mode)) {
	  int len;
	  char symlink[MAXPATHLEN];

	  if (symlink_count > MAXSYMLINKS) {
	    DPRINTF((stderr, "Symbolic link limit exceeded\n"));
	    errno = EMLINK;
	    return NULL;
	  }

	  DPRINTF((stderr, "Symbolic link detected - %s\n", result));
	  len = readlink(result, symlink, sizeof(symlink));
	  symlink[len] = '\0';

	  /* If the symlink isn't absolute, back off our last addition,
	   * since the symlink is interpreted relative to the directory
	   * it's in.
	   */

	  if (symlink[0] != '/')  p = savedp;

	  p = expand_path(symlink, symlink_flag, result, p, symlink_count+1);
	}
      }
    }
  }

  /* Special case - if we've seen enough ..'s, then we're back at
   * the beginning of the buffer.  Make sure that at least the first
   * slash is present.
   */

  if (p == result) {
    p ++;
  }

  return p;
}

/* char * resolve_path (char *result, const char *pathname, symlink_flag)
 *
 * Our main workhorse function, called by all the overridden functions,
 * which converts a (potentially) relative pathname into an absolute
 * path and returns that result.
 *
 * "result" should point to a char[MAXPATHLEN] array
 *
 * WARNING: Doesn't check for buffer overflows :-(
 */

static const char * resolve_path (char *result, const char *pathname,
				  enum symlink_flag symlink_flag)
{
  char *p;

  if (!enable_module) {
    DPRINTF((stderr, "cwd: module disabled; resolve_path does nothing\n"));
    return pathname;
  }

  DPRINTF((stderr, "Entering resolve_path('%s')\n", pathname));

  if (pathname == NULL) {
    DPRINTF((stderr, "resolve_path('%s') returns NULL\n", pathname));
    return NULL;
  }
  if (pathname[0] == '\0') {
    DPRINTF((stderr, "resolve_path('') returns ''\n"));
    return pathname;
  }

  p = result;
  if (pathname[0] != '/') {
    strcpy(result, cwd);
    for (p=result; *p; p++);
    *(p++) = '/';
  }

  enable_module = 0;
  p = expand_path(pathname, symlink_flag, result, p, 0);
  enable_module = 1;

  if (p) {
    *p = '\0';
  } else {
    result = NULL;
  }

  DPRINTF((stderr, "resolve_path returns '%s'\n", result));
  return result;
}

int chdir(const char *path)
{
  struct stat statstruct;
  char result[MAXPATHLEN];

  if (!resolve_path(result, path, follow_symlinks)) return -1;
  enable_module = 0;
  if (stat(result, &statstruct) == 0) {
    if (S_ISDIR(statstruct.st_mode)) {
      DPRINTF((stderr, "cwd %d: chdir(`%s')\n", getpid(), result));
      strcpy(cwd, result);
      enable_module = 1;
      return 0;
    } else {
      errno = ENOTDIR;
    }
  }
  enable_module = 1;
  return -1;
}

__ALIAS(int, chdir, (const char *));

static char * fdarray = NULL;
static int fdarray_size = 0;

static int (*nextopen)(const char *, int, ...);

int open (const char *pathname, int flags, ...)
{
  va_list ap;
  int mode;
  int retval;
  char scratch[MAXPATHLEN];
  const char *result;

  va_start(ap, flags);
  mode = va_arg(ap, int);
  va_end(ap);

  if (nextopen == NULL) {
    nextopen = dlsym(RTLD_NEXT, "open");
  }

  DPRINTF((stderr, "cwd %d: Entering open(`%s')\n", getpid(), pathname));

  result = resolve_path(scratch, pathname, follow_symlinks);
  if (!result) return -1;
  retval = (*nextopen)(result, flags, mode);

  /* If the open succeeded, we need to save the pathname, in case
   * of a later fchdir()
   */

  if (retval != -1) {
    if (fdarray_size == 0) {
      fdarray = malloc((retval+1) * MAXPATHLEN);
      if (fdarray == NULL) {
	fprintf(stderr, "Malloc failed in cwd:open()\n");
	return retval;
      }
      fdarray_size = retval+1;
    } else if (fdarray_size < retval+1) {
      fdarray = realloc(fdarray, (retval+1) * MAXPATHLEN);
      if (fdarray == NULL) {
	fprintf(stderr, "Malloc failed in cwd:open()\n");
	return retval;
      }
      fdarray_size = retval+1;
    }
    strcpy(fdarray + retval*MAXPATHLEN, result);
  }

  return retval;
}

__ALIAS(int, open, (const char *, int flags, ...));

int fchdir(int fd)
{
  struct stat statstruct;
  char result[MAXPATHLEN];

  if (!resolve_path(result, fdarray + fd*MAXPATHLEN, follow_symlinks)) {
    return -1;
  }
  enable_module = 0;
  if (stat(result, &statstruct) == 0) {
    if (S_ISDIR(statstruct.st_mode)) {
      DPRINTF((stderr, "cwd %d: fchdir(`%s')\n", getpid(), result));
      strcpy(cwd, result);
      enable_module = 1;
      return 0;
    } else {
      errno = ENOTDIR;
    }
  }
  enable_module = 1;
  return -1;
}

__ALIAS(int, fchdir, (int));

/* execve() is a bit tricky, since we can't depend on the kernel anymore
 * to pass current working directory to the child process.  We pass it
 * ourselves by setting "PWD" in the environment.  This code ties in
 * with the _init() code (in this file) that pulls cwd out of PWD as the
 * new program starts up.
 */

static int (*nextexecve)(const char *filename, char *const argv [],
			 char *const envp[]);

int execve(const char *filename, char *const argv [], char *const envp[])
{
  int envp_entries;
  int pwd_entry = -1;
  char result[MAXPATHLEN];
  char PWDenv[MAXPATHLEN+5];
  char **new_envp;

  DPRINTF((stderr, "cwd: Entering execve; path='%s'\n", filename));

  if (nextexecve == NULL) {
    nextexecve = dlsym(RTLD_NEXT, "execve");
  }

  strcpy(PWDenv, "PWD=");
  strcpy(PWDenv+4, cwd);

  /* Count the size of the original environment, noting the location
   * of a PWD entry, if one exists
   */

  for (envp_entries=0; envp[envp_entries]; envp_entries++) {
    if (strncmp(envp[envp_entries], "PWD=", 4) == 0) pwd_entry = envp_entries;
  }

  /* Allocate a new envp array, making room for both the NULL and a
   * extra variable, in case we need to add a PWD entry (instead of
   * replacing an existing one)
   */

  new_envp = malloc((envp_entries+2) * sizeof(char *));
  if (new_envp == NULL) {
    errno = ENOMEM;
    return -1;
  }

  memcpy(new_envp, envp, (envp_entries+1) * sizeof(char *));

  if (pwd_entry == -1) {
    new_envp[envp_entries++] = PWDenv;
    new_envp[envp_entries] = NULL;
  } else {
    new_envp[pwd_entry] = PWDenv;
  }

  if (!resolve_path(result, filename, follow_symlinks)) {
    return -1;
  }
  return (*nextexecve)(result, argv, new_envp);
}

#ifdef __OVERRIDE
int __execve(const char *filename, char *const argv [],
	     char *const envp[]) __attribute__ ((alias("execve")));
#endif


#ifdef OVERRIDE_getcwd

char * getcwd(char *buffer, size_t size) {
  int cwdlen = strlen(cwd);

  if (cwdlen+1 > size) {
    errno = ERANGE;
    return NULL;
  }

  if (buffer == NULL) {
    if (size >= 0) {
      buffer = (char *) malloc(size);
    } else {
      buffer = (char *) malloc(cwdlen+1);
    }
    if (buffer == NULL) {
      return NULL;
    }
  }

  strcpy(buffer, cwd);
  return buffer;
}

#endif

/* The remaining functions that need to be overridden all follow a
 * standard format.  The only thing we need to do is toss in one
 * or two calls to resolve_path() before passing the results on to
 * the original function.  The macro makes things a bit simplier,
 * at some expense in readability...
 */

#define OVERRIDE_SYSCALL(type, syscall, arglist, varlist, SYMLINK_FLAG, eret) \
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  const char *errval;							\
  char result[MAXPATHLEN];						\
  type retval;								\
  if (next##syscall == NULL) {						\
    next##syscall = dlsym(RTLD_NEXT, #syscall);				\
    if ((errval = dlerror()) != NULL) {					\
      fprintf(stderr, "dlsym(" #syscall "): %s\n", errval);		\
    }									\
  }									\
  DPRINTF((stderr, "cwd: Entering " #syscall "; path='%s'\n", path));	\
  path = resolve_path(result, path, SYMLINK_FLAG);			\
  if (! path) {								\
    return eret;							\
  }									\
  retval = (*next##syscall) varlist;					\
  DPRINTF((stderr, "cwd: " #syscall " returning %d\n", retval));	\
  return retval;							\
}									\
__ALIAS(type, syscall, arglist)

#define OVERRIDE_SYSCALL2(type, syscall, arglist, varlist, SYMLINK_FLAG, SYMLINK2_FLAG) \
static type (*next##syscall) arglist = NULL;				\
type syscall arglist {							\
  const char *errval;							\
  char result[MAXPATHLEN];						\
  char result2[MAXPATHLEN];						\
  if (next##syscall == NULL) {						\
    next##syscall = dlsym(RTLD_NEXT, #syscall);				\
    if ((errval = dlerror()) != NULL) {					\
      fprintf(stderr, "dlsym(" #syscall "): %s\n", errval);		\
    }									\
  }									\
  DPRINTF((stderr, "cwd: Entering " #syscall "; path='%s'\n", path));	\
  path = resolve_path(result, path, SYMLINK_FLAG);			\
  path2 = resolve_path(result2, path2, SYMLINK2_FLAG);			\
  if (! path || ! path2) {						\
    return -1;								\
  }									\
  return (*next##syscall) varlist;					\
}									\
__ALIAS(type, syscall, arglist)


OVERRIDE_SYSCALL(int, xstat, (int ver, const char *path, struct stat *buf), (ver, path, buf), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, lxstat, (int ver, const char *path, struct stat *buf), (ver, path, buf), show_last_symlink, -1);
OVERRIDE_SYSCALL(int, xstat64, (int ver, const char *path, struct stat64 *buf), (ver, path, buf), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, lxstat64, (int ver, const char *path, struct stat64 *buf), (ver, path, buf), show_last_symlink, -1);
OVERRIDE_SYSCALL(int, xmknod, (int ver, const char *path, mode_t mode, dev_t *dev), (ver, path, mode, dev), follow_symlinks, -1);

OVERRIDE_SYSCALL(int, access, (const char *path, int mode), (path, mode), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, acct, (const char *path), (path), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, chmod, (const char *path, mode_t mode), (path, mode), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, chown, (const char *path, uid_t uid, gid_t gid), (path, uid, gid), follow_symlinks, -1);
OVERRIDE_SYSCALL2(int, link, (const char *path, const char *path2), (path, path2), follow_symlinks, follow_symlinks);
OVERRIDE_SYSCALL(int, unlink, (const char *path), (path), show_last_symlink, -1);
OVERRIDE_SYSCALL(int, symlink, (const char *symlink, const char *path), (symlink, path), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, readlink, (const char *path, char *buf, size_t bufsiz), (path, buf, bufsiz), show_last_symlink, -1);
OVERRIDE_SYSCALL2(int, rename, (const char *path, const char *path2), (path, path2), show_last_symlink, show_last_symlink);
OVERRIDE_SYSCALL(int, mkdir, (const char *path, mode_t mode), (path, mode), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, rmdir, (const char *path), (path), follow_symlinks, -1);
OVERRIDE_SYSCALL(int, utime, (const char *path, const struct utimbuf *buf), (path, buf), follow_symlinks, -1);

OVERRIDE_SYSCALL(DIR *, opendir, (const char *path), (path), follow_symlinks, NULL);

void _init(void)
{
  char *pwd;
  const char *errval;
  char * (*nextgetcwd)(char *buffer, size_t size);

  DPRINTF((stderr, "cwd: GotInit\n"));

  /* These three functions (xstat, lxstat, xmknod) are special because
   * they only exist in "__" form.  The user-visible versions are
   * actually inline functions in the include files.
   */

  nextxstat = dlsym(RTLD_NEXT,"__xstat");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(__xstat): %s\n", errval);
  }

  nextlxstat = dlsym(RTLD_NEXT,"__lxstat");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(__lxstat): %s\n", errval);
  }

  nextxmknod = dlsym(RTLD_NEXT,"__xmknod");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(__xmknod): %s\n", errval);
  }



  nextgetcwd = dlsym(RTLD_NEXT,"getcwd");
  if ((errval = dlerror()) != NULL) {
    fprintf(stderr, "dlsym(getcwd): %s\n", errval);
  }

  if ((pwd = getenv("PWD")) != NULL) {
    strcpy(cwd, pwd);
    DPRINTF((stderr, "cwd = '%s' (via PWD)\n", cwd));
  } else if (!nextgetcwd || ((*nextgetcwd)(cwd, sizeof(cwd)) == NULL)) {
    if (!nextgetcwd) fprintf(stderr, "No nextgetcwd; ");
    fprintf(stderr, "getcwd failed; cwd is %s; errno = %d\n", cwd, errno);
    cwd[0]='/';
    cwd[1]='\0';
  } else {
    DPRINTF((stderr, "cwd = '%s' (via getcwd)\n", cwd));
  }

  enable_module = 1;

  DPRINTF((stderr, "cwd: EndInit\n"));
}
